use strict;
use warnings;

package Jifty::Plugin::Comment::Mixin::Model::Commented;
use base qw/ Jifty::DBI::Record::Plugin /;

our @EXPORT = qw( comments comment_record_class for_commenting );

use Scalar::Util qw/ blessed /;

use Jifty::DBI::Schema;
use Jifty::Record schema {
};

=head1 NAME

Jifty::Plugin::Comment::Mixin::Model::Commented - add comments to a model

=head1 SYNOPSIS

  package App::Model::Fooble;

  use Jifty::DBI::Schema;
  use App::Record schema {
      column scribble => type is 'text';
      column wobble => type is 'int';
  };

  use Jifty::Plugin::Comment::Mixin::Model::Commented;

=head1 DESCRIPTION

Add this mixin to a model if you'd like to attach comments to it. Comments can be used to allow users of your system to comment upon and discuss the record to which they are attached.

=head1 METHODS

=head2 import

This method performs some rather devious magic to make everything work easily. It automatically generates an additional model for your application. This model will look something like this:

  use strict;
  use warnings;
  
  package App::Model::FoobleComment;
  use Jifty::DBI::Schema;
  
  use Jifty::Record schema {
      column commented_upon =>
          references App::Model::Fooble,
          label is 'Commented upon',
          is mandatory,
          is immutable,
          ;
  
      column the_comment =>
          references App::Model::Comment,
          label is 'Comment',
          is mandatory,
          is immutable,
          is distinct,
          ;
  };

  App::Model::FoobleComment->add_trigger( before_access => sub {
      my $self = shift;
      my ($right, %args) = @_;

      if ($right eq 'create') {
          return 'allow' if $self->current_user->id;
      }

      if ($right eq 'read') {
          return 'allow';
      }

      return $self->App::Model::FoobleComment::current_user_can(@_);
  });
  
You will need to define an C<before_access> trigger for this class if you want it to be useful.

=cut

sub import {
    my $my_class      = caller;
    my $comment_class = $my_class.'Comment';
    my $app_class     = Jifty->app_class;

    eval "
use strict;
use warnings;

package ${comment_class};
use Jifty::DBI::Schema;

use Jifty::Record schema {
    column commented_upon =>
        references ${my_class},
        label is 'Commented upon',
        is mandatory,
        is immutable,
        ;

    column the_comment =>
        references ${app_class}::Model::Comment,
        label is 'Comment',
        is mandatory,
        is immutable,
        is distinct,
        ;
};

1;
";

    die "Failed to create comment link model ${comment_class}: $@" if $@;

    Jifty->class_loader->_require_model_related_classes($comment_class);

    my $comment_filename = $comment_class;
    $comment_filename =~ s{::}{/}g;
    $comment_filename .= '.pm';
    $INC{ $comment_filename } = 'autogenerated';

    goto &Jifty::DBI::Record::Plugin::import;
}

=head2 for_commenting

Returns a value to be used with the comment views. It's basically just a string identifying the class name and ID of the record.

=cut

sub for_commenting {
    my $self = shift;
    return blessed($self) . '-' . $self->id;
}

=head2 comments

Returns a collection of L<Jifty::Plugin::Comment::Model::Comment> objects that have been attached to the current record. (Actually, it returns the a collection of the local application class, e.g. C<App::Model::CommentCollection>.)

=cut

sub comments {
    my $self = shift;

    my $comments = Jifty->app_class('Model', 'CommentCollection')->new;
    my $link_alias = $comments->join(
        column1 => 'id',
        table2  => $self->comment_record_class->table,
        column2 => 'the_comment',
    );

    $comments->limit(
        alias  => $link_alias,
        column => 'commented_upon',
        value  => $self->id,
    );

    return $comments;
}

=head2 comment_record_class

This is the name of the linking class that was created during L</import>.

=cut

sub comment_record_class {
    my $self = shift;
    return (ref $self || $self).'Comment';
}

=head1 AUTHOR

Andrew Sterling Hanenkamp C<< <hanenkamp@cpan.com> >>

=head1 COPYRIGHT AND LICENSE

Copyright 2007 Boomer Consulting, Inc. All Rights Reserved.

This program is free software and may be modified and distributed under the same terms as Perl itself.

=cut

1;

